<?php

declare(strict_types=1);

namespace Erlage\Photogram\Data\Models\Post;

use Erlage\Photogram\Settings;
use Erlage\Photogram\Constants\SystemConstants;
use Erlage\Photogram\Constants\ResponseConstants;
use Erlage\Photogram\Data\Dtos\Post\PostMetaHashtagsDTO;
use Erlage\Photogram\Reflection\TraitReflectionClassName;
use Erlage\Photogram\Data\Dtos\Post\PostDisplayContentDTO;

final class PostValidator
{
    use TraitReflectionClassName;

    public static function displayCaption(string $displayCaption): string
    {
        if (\strlen($displayCaption) > Settings::getInt(SystemConstants::SETTING_MAX_LEN_POST_DISPLAY_CAPTION))
        {
            return ResponseConstants::D_ERROR_POST_DISPLAY_CAPTION_MAX_LEN_MSG;
        }

        if (\strlen($displayCaption) < Settings::getInt(SystemConstants::SETTING_MIN_LEN_POST_DISPLAY_CAPTION))
        {
            return ResponseConstants::D_ERROR_POST_DISPLAY_CAPTION_MIN_LEN_MSG;
        }

        return SystemConstants::OK;
    }

    public static function displayLocation(string $displayLocation): string
    {
        if (\strlen($displayLocation) > Settings::getInt(SystemConstants::SETTING_MAX_LEN_POST_DISPLAY_LOCATION))
        {
            return ResponseConstants::D_ERROR_POST_DISPLAY_LOCATION_MAX_LEN_MSG;
        }

        if (\strlen($displayLocation) < Settings::getInt(SystemConstants::SETTING_MIN_LEN_POST_DISPLAY_LOCATION))
        {
            return ResponseConstants::D_ERROR_POST_DISPLAY_LOCATION_MIN_LEN_MSG;
        }

        return SystemConstants::OK;
    }

    public static function displayContent(PostDisplayContentDTO $displayContent): string
    {
        /*
        |--------------------------------------------------------------------------
        | number of images/videos
        |--------------------------------------------------------------------------
        */

        $total = $displayContent -> getNumberOfItems();

        if ($total > Settings::getInt(SystemConstants::SETTING_MAX_POST_DISPLAY_CONTENT_ITEM))
        {
            return ResponseConstants::D_ERROR_POST_DISPLAY_CONTENT_ITEM_MAX_MSG;
        }

        if ($total < Settings::getInt(SystemConstants::SETTING_MIN_POST_DISPLAY_CONTENT_ITEM))
        {
            return ResponseConstants::D_ERROR_POST_DISPLAY_CONTENT_ITEM_MIN_MSG;
        }

        /*
        |--------------------------------------------------------------------------
        | number of users tagged
        |--------------------------------------------------------------------------
        */

        $displayContent -> iteratorPrepare();

        $totalNumberOfUsersTagged = 0;

        while ($displayContent -> iteratorForward())
        {
            $displayContentItemDTO = $displayContent -> iteratorCurrentDisplayContentItem();

            $numberOfUsersTagggedInCurrentItem = $displayContentItemDTO -> getNumberOfUsersTagged();

            if (
                $numberOfUsersTagggedInCurrentItem > Settings::getInt(
                    SystemConstants::SETTING_MAX_POST_USER_TAG_PER_ITEM
                )
            ) {
                return ResponseConstants::D_ERROR_POST_USER_TAG_PER_ITEM_MAX;
            }

            if (
                $numberOfUsersTagggedInCurrentItem < Settings::getInt(
                    SystemConstants::SETTING_MIN_POST_USER_TAG_PER_ITEM
                )
            ) {
                return ResponseConstants::D_ERROR_POST_USER_TAG_PER_ITEM_MIN;
            }

            if ($numberOfUsersTagggedInCurrentItem > 0)
            {
                $totalNumberOfUsersTagged += $numberOfUsersTagggedInCurrentItem;
            }
        }

        if ($totalNumberOfUsersTagged > Settings::getInt(SystemConstants::SETTING_MAX_POST_USER_TAG_TOTAL))
        {
            return ResponseConstants::D_ERROR_POST_USER_TAG_TOTAL_MAX;
        }

        if ($totalNumberOfUsersTagged < Settings::getInt(SystemConstants::SETTING_MIN_POST_USER_TAG_TOTAL))
        {
            return ResponseConstants::D_ERROR_POST_USER_TAG_TOTAL_MIN;
        }

        return SystemConstants::OK;
    }

    public static function metaHashtags(PostMetaHashtagsDTO $metaHashtags): string
    {
        $total = $metaHashtags -> getLength();

        if ($total > Settings::getInt(SystemConstants::SETTING_MAX_POST_HASHTAG))
        {
            return ResponseConstants::D_ERROR_POST_HASHTAG_MAX_MSG;
        }

        if ($total < Settings::getInt(SystemConstants::SETTING_MIN_POST_HASHTAG))
        {
            return ResponseConstants::D_ERROR_POST_HASHTAG_MIN_MSG;
        }

        $metaHashtags -> iteratorPrepare();

        while ($metaHashtags -> iteratorForward())
        {
            $hashtagLiteral = $metaHashtags -> iteratorCurrentHashtagLiteral();

            if (\strlen($hashtagLiteral) < Settings::getInt(SystemConstants::SETTING_MIN_LEN_POST_HASHTAG))
            {
                return ResponseConstants::D_ERROR_POST_HASHTAG_MIN_LEN_MSG;
            }

            if (\strlen($hashtagLiteral) > Settings::getInt(SystemConstants::SETTING_MAX_LEN_POST_HASHTAG))
            {
                return ResponseConstants::D_ERROR_POST_HASHTAG_MAX_LEN_MSG;
            }
        }

        return SystemConstants::OK;
    }
}
